<?php

namespace App\Imports;

use App\Models\User;
use App\Models\Siswa;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithBatchInserts;
use Maatwebsite\Excel\Concerns\WithChunkReading;

class SiswaImport implements ToCollection, WithHeadingRow, WithBatchInserts, WithChunkReading
{
    protected $skipNisn = [];

    public function collection(Collection $rows)
    {
        $this->validateEmptyColumns($rows);

        // Cek duplikat Excel + DB
        $this->validateDuplicateNisnExcelOnly($rows);
        $this->skipNisn = $this->getExistingNisns($rows);

        // Jika semua data NISN sudah ada di sistem, error
        if (count($this->skipNisn) === count($rows)) {
            throw new \Exception("Semua data NISN sudah terdaftar di sistem. Tidak ada data yang dapat diimport.");
        }

        DB::beginTransaction();

        try {
            foreach ($rows as $row) {
                $nisn = trim($row['nisn']);
                $email = trim($row['email']);

                if (in_array($nisn, $this->skipNisn)) {
                    continue;
                }

                // Cek apakah email sudah ada
                if (User::where('email', $email)->exists()) {
                    continue;
                }

                $password = !empty($row['password']) ? $row['password'] : '12345678';

                $user = User::create([
                    'name'     => trim($row['name']),
                    'email'    => $email,
                    'password' => Hash::make($password),
                    'role'     => 'siswa',
                ]);

                Siswa::create([
                    'siswa_id' => $user->id,
                    'nisn'     => $nisn,
                ]);

                Log::info('Berhasil import siswa', [
                    'name'  => $row['name'],
                    'email' => $email,
                    'nisn'  => $nisn
                ]);
            }

            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Import siswa gagal', [
                'message' => $e->getMessage(),
                'trace'   => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    private function validateEmptyColumns(Collection $rows)
    {
        $emptyColumns = [];
        $rowNumber = 2;

        foreach ($rows as $row) {
            $missingFields = [];

            if (empty($row['name']) || trim($row['name']) === '') {
                $missingFields[] = 'name';
            }

            if (empty($row['email']) || trim($row['email']) === '') {
                $missingFields[] = 'email';
            }

            if (empty($row['nisn']) || trim($row['nisn']) === '') {
                $missingFields[] = 'nisn';
            }

            if (!empty($missingFields)) {
                $emptyColumns[] = "Baris {$rowNumber}: kolom " . implode(', ', $missingFields) . " kosong";
            }

            $rowNumber++;
        }

        if (!empty($emptyColumns)) {
            throw new \Exception("Silakan isi data kolom yang kosong: " . implode('; ', $emptyColumns));
        }
    }

    private function validateDuplicateNisnExcelOnly(Collection $rows)
    {
        $nisnTracker = [];
        $duplicateInExcel = [];
        $rowNumber = 2;

        foreach ($rows as $row) {
            $nisn = trim($row['nisn']);

            if (isset($nisnTracker[$nisn])) {
                $duplicateInExcel[] = "NISN {$nisn} duplikat antara baris {$nisnTracker[$nisn]} dan baris {$rowNumber}";
            } else {
                $nisnTracker[$nisn] = $rowNumber;
            }

            $rowNumber++;
        }

        if (!empty($duplicateInExcel)) {
            throw new \Exception("Data NISN duplikat dalam Excel: " . implode('; ', $duplicateInExcel) . ". Silakan perbaiki file Excel Anda.");
        }
    }

    private function getExistingNisns(Collection $rows): array
    {
        $nisns = $rows->pluck('nisn')->map(fn($n) => trim($n))->unique()->toArray();
        return Siswa::whereIn('nisn', $nisns)->pluck('nisn')->toArray();
    }

    public function batchSize(): int
    {
        return 100;
    }

    public function chunkSize(): int
    {
        return 100;
    }
}
