<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Guru;
use App\Models\Siswa;
use App\Models\TahunAjaran;
use App\Models\Kelas;
use App\Models\KelasMapel;
use App\Models\SiswaKelas;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\JadwalLayanan;
use App\Models\LayananKonseling;
use App\Models\LayananParenting;
use App\Models\KehadiranSiswa;

class SiswaController extends Controller
{

public function index()
{
    $user = Auth::user();

    if ($user->role !== 'siswa') {
        abort(403);
    }

    $siswa = $user->siswa;

    // Ambil semua kelas yang pernah diikuti siswa
    $kelasIds = SiswaKelas::where('siswa_id', $siswa->siswa_id)
        ->pluck('kelas_id');

    // Jika siswa belum memiliki kelas, return view dengan data kosong
    if ($kelasIds->isEmpty()) {
        return view('backend.siswa.index', [
            'siswa' => $siswa,
            'kelases' => collect([]),
            'kehadiranPerKelas' => collect([])
        ]);
    }

    // Ambil semua kelas lengkap dengan tahun ajaran
    $allKelas = Kelas::whereIn('id', $kelasIds)
        ->with('tahunAjaran')
        ->get();

    // Jika tidak ada kelas yang ditemukan, return view dengan data kosong
    if ($allKelas->isEmpty()) {
        return view('backend.siswa.index', [
            'siswa' => $siswa,
            'kelases' => collect([]),
            'kehadiranPerKelas' => collect([])
        ]);
    }

    // Urutkan kelas berdasarkan tahun ajaran terbaru
    $sortedKelas = $allKelas->sortByDesc(function ($kelas) {
        return $kelas->tahunAjaran->tahun_awal ?? 0;
    });

    // Ambil ID tahun ajaran terbaru dengan pengecekan null
    $kelasWithTahunAjaran = $sortedKelas->first();
    $tahunAjaranTerbaru = $kelasWithTahunAjaran && $kelasWithTahunAjaran->tahunAjaran 
        ? $kelasWithTahunAjaran->tahunAjaran->tahun_awal 
        : null;

    // Jika tidak ada tahun ajaran terbaru, return view dengan data kosong
    if (!$tahunAjaranTerbaru) {
        return view('backend.siswa.index', [
            'siswa' => $siswa,
            'kelases' => collect([]),
            'kehadiranPerKelas' => collect([])
        ]);
    }

    // Ambil kelas-kelas untuk tahun ajaran terbaru
    $kelases = Kelas::with([
        'semester',
        'tahunAjaran',
        'kelasMapel.mapel',
        'kelasMapel.guru.user',
        'kelasMapel.nilai' => function($q) use ($siswa) {
            $q->where('siswa_id', $siswa->siswa_id);
        }
    ])
    ->whereIn('id', $kelasIds)
    ->whereHas('tahunAjaran', function($q) use ($tahunAjaranTerbaru) {
        $q->where('tahun_awal', $tahunAjaranTerbaru);
    })
    ->orderByDesc('semester_id')
    ->get();

    // Ambil data kehadiran per kelas
    $kehadiranPerKelas = KehadiranSiswa::with('bulan')
        ->where('siswa_id', $siswa->siswa_id)
        ->get()
        ->groupBy(fn($k) => $k->bulan->kelas_id)
        ->map(function($items) {
            return [
                'sakit' => $items->sum('sakit'),
                'izin' => $items->sum('izin'),
                'alpha' => $items->sum('alpha'),
            ];
        });

    return view('backend.siswa.index', [
        'siswa' => $siswa,
        'kelases' => $kelases,
        'kehadiranPerKelas' => $kehadiranPerKelas
    ]);
}


public function profil()
{
    $user = Auth::user();

    if ($user->role !== 'siswa') {
        abort(403);
    }

    $siswa = $user->siswa;

    // Ambil tahun ajaran terbaru
    $tahunAjaranTerbaru = TahunAjaran::orderByDesc('tahun_awal')->first();

    // Ambil kelas aktif siswa di tahun ajaran terbaru
    $kelasAktif = null;
    if ($tahunAjaranTerbaru && $siswa) {
        $kelasAktif = Kelas::whereHas('semester', function ($q) use ($tahunAjaranTerbaru) {
                $q->where('tahun_ajaran_id', $tahunAjaranTerbaru->id);
            })
            ->whereHas('siswaKelas', function ($q) use ($siswa) {
                $q->where('siswa_id', $siswa->siswa_id);
            })
            ->with(['semester.tahunAjaran'])
            ->first();
    }

    return view('backend.siswa.profil', compact('siswa', 'kelasAktif'));
}

public function updateSiswa(Request $request, $id)
{
    $siswa = Siswa::findOrFail($id);

    // Ambil semua field yang diperlukan
    $data = $request->only([
        'nisn',
        'alamat',
        'tempat_lahir',
        'tanggal_lahir',
        'jenis_kelamin',
        'agama',
        'no_hp',
        'sekolah_asal',
        'lintang',
        'bujur',
        'jenis_tinggal',
        'berkebutuhan_khusus',
        'anak_ke',
        'jumlah_saudara_kandung',
        'nipd',
        'alat_transportasi',
        'no_telepon',
        'rombel_saat_ini',
        'jarak_rumah',
        'data_berubah',
        'deskripsi',
    ]);

    $data['alamat_lengkap']      = $request->input('alamat_lengkap', []);
    $data['bantuan_sosial']      = $request->input('bantuan_sosial', []);
    $data['data_ayah']           = $request->input('data_ayah', []);
    $data['data_ibu']            = $request->input('data_ibu', []);
    $data['data_wali']           = $request->input('data_wali', []);
    $data['data_fisik']          = $request->input('data_fisik', []);
    $data['data_rekening']       = $request->input('data_rekening', []);
    $data['dokumen_identitas']   = $request->input('dokumen_identitas', []);

    $siswa->update($data);

    return back()->with('success','Data siswa berhasil diperbarui');
}

public function nilai(Request $request)
{
    $siswa = Auth::user()->siswa;
    $kelasParam = $request->kelas;

    // Ambil semua kelas yg pernah diikuti siswa
    $kelasIds = SiswaKelas::where('siswa_id', $siswa->siswa_id)
                  ->pluck('kelas_id');

    $allKelas = Kelas::whereIn('id', $kelasIds)
        ->with('tahunAjaran')
        ->get();

    // Dropdown kelas
    $dropdownKelas = $allKelas->unique(function ($k) {
        return $k->nama . '-' . $k->tahun_ajaran_id;
    })->sortByDesc(fn($k) => $k->tahunAjaran->tahun_awal ?? 0);

    $kelases = collect();
    $selectedKey = null;

    if ($kelasParam) {
        if (is_numeric($kelasParam)) {
            $kelas = Kelas::find($kelasParam);
            $nama = $kelas->nama;
            $tahunAjaranId = $kelas->tahun_ajaran_id;
            $selectedKey = $nama . '-' . $tahunAjaranId;
        } else {
            [$nama, $tahunAjaranId] = explode('-', $kelasParam);
            $selectedKey = $kelasParam;
        }

        $kelases = Kelas::with([
            'semester',
            'tahunAjaran',
            'kelasMapel' => function($q) use ($siswa) {
                $q->with([
                    'mapel',
                    'guru.user',
                    'nilai' => fn($q2) => $q2->where('siswa_id', $siswa->siswa_id),
                ]);
            },
        ])
        ->where('nama', $nama)
        ->where('tahun_ajaran_id', $tahunAjaranId)
        ->whereIn('id', $kelasIds)
        ->orderByDesc('semester_id')
        ->get();
    }

    return view('backend.siswa.nilai', compact(
        'kelases', 'siswa', 'dropdownKelas', 'selectedKey'
    ));
}

public function layanan()
{
    $siswa = Auth::user()->siswa;

    // Ambil semua guru kecuali kepala_sekolah
    $gurus = Guru::with('user')
        ->whereHas('user', function ($query) {
            $query->where('role', '!=', 'kepala_sekolah');
        })->get();

    $layanans = LayananKonseling::with(['jadwal.guru' => function ($q) {
        $q->withTrashed();
    }, 'jadwal.guru.user'])
    ->where('siswa_id', $siswa->siswa_id)
    ->latest()
    ->get();

    return view('backend.siswa.layanan', compact('gurus', 'layanans'));
}


public function getJadwalGuru($guru_id)
{
    $jadwals = JadwalLayanan::where('guru_id', $guru_id)
        ->with(['layanan' => function ($query) {
            $query->whereIn('status', ['Diproses', 'Disetujui']);
        }])
        ->get();

    $result = $jadwals->map(function ($j) {
        $unavailable = $j->layanan->isNotEmpty();

        return [
            'id'          => $j->id,
            'hari'        => $j->hari,
            'jam'         => $j->jam,
            'unavailable' => $unavailable,
        ];
    });

    return response()->json($result);
}
public function storeLayanan(Request $request)
{
    $request->validate([
        'guru_id'    => 'required|exists:guru,guru_id',
        'jadwal_id'  => 'required|exists:jadwal_layanans,id',
        'tanggal'    => 'required|date',
        'keterangan' => 'required|string',
    ]);

    $siswa = Auth::user()->siswa;
    $jadwal = JadwalLayanan::with('guru.user')->findOrFail($request->jadwal_id);
    $guru = $jadwal->guru;

    LayananKonseling::create([
        'jadwal_layanan_id' => $jadwal->id,
        'siswa_id'          => $siswa->siswa_id,
        'keterangan'        => $request->keterangan,
        'tanggal'           => $request->tanggal,
        'status'            => 'Diproses',
    ]);

    // Tentukan waktu salam
    $jamSekarang = now()->format('H');
    if ($jamSekarang >= 5 && $jamSekarang < 11) {
        $salam = 'Selamat pagi';
    } elseif ($jamSekarang >= 11 && $jamSekarang < 15) {
        $salam = 'Selamat siang';
    } elseif ($jamSekarang >= 15  && $jamSekarang < 18) {
        $salam = 'Selamat sore';
    } else {
        $salam = 'Selamat malam';
    }

    // Tentukan sapaan berdasarkan jenis kelamin
    $sapaan = $guru->jenis_kelamin === 'L' ? 'Pak' : 'Bu';

    // Nomor HP dengan format +62
    $noHpGuru = preg_replace('/[^0-9]/', '', $guru->no_hp);
    if (substr($noHpGuru, 0, 1) === '0') {
        $noHpGuru = '62' . substr($noHpGuru, 1);
    }

    $kelas = data_get($siswa, 'kelas.nama', 'tidak diketahui');

    // Format pesan
    $pesan = "{$salam} {$sapaan} {$guru->user->name}, saya {$siswa->user->name} dari kelas {$kelas} ingin melakukan layanan konseling di:\n\n"
        . "Hari: {$jadwal->hari}\n"
        . "Jam: {$jadwal->jam}\n"
        . "Tanggal: {$request->tanggal}\n"
        . "Keterangan: {$request->keterangan}\n\n"
        . "Mohon konfirmasinya {$sapaan}, terima kasih";

    $urlWhatsapp = 'https://wa.me/' . $noHpGuru . '?text=' . urlencode($pesan);

    return redirect()->route('siswa.layanan')
    ->with('wa_url', $urlWhatsapp);

}

public function layananParenting()
{
    $siswa = Auth::user();
    $tahunAjaran = TahunAjaran::orderByDesc('tahun_awal')->first();

    $kelasId = SiswaKelas::where('siswa_id', $siswa->id)
        ->whereHas('kelas', function ($q) use ($tahunAjaran) {
            $q->where('tahun_ajaran_id', $tahunAjaran->id);
        })->pluck('kelas_id')->first();

    $guruIds = KelasMapel::where('kelas_id', $kelasId)->pluck('guru_id');

    $guruBk = Guru::withTrashed()
        ->whereIn('guru_id', $guruIds)
        ->whereHas('user', function ($q) {
            $q->where('role', 'guru_bk');
        })
        ->with('user')
        ->first();

    $layanans = LayananParenting::where('siswa_id', $siswa->id)
        ->where('tipe', 1)
        ->latest()
        ->get();

    return view('backend.siswa.layananParenting', [
        'layanans' => $layanans,
        'guruBk'   => $guruBk
    ]);
}


public function store(Request $request)
{
    $request->validate([
        'permasalahan' => 'required|string',
        'tanggal'      => 'required|date',
    ]);

    $siswa = Auth::user();

    // Ambil tahun ajaran aktif
    $tahunAjaran = TahunAjaran::orderByDesc('tahun_awal')->first();

    // Cari kelas siswa
    $kelasId = SiswaKelas::where('siswa_id', $siswa->id)
                ->whereHas('kelas', function ($q) use ($tahunAjaran) {
                    $q->where('tahun_ajaran_id', $tahunAjaran->id);
                })->pluck('kelas_id')->first();

    // Cari guru_id dari kelas_mapel
    $guruIds = KelasMapel::where('kelas_id', $kelasId)->pluck('guru_id');

    // Ambil guru yang role-nya 'guru_bk'
    $guruBk = Guru::whereIn('guru_id', $guruIds)
                ->whereHas('user', function ($q) {
                    $q->where('role', 'guru_bk');
                })->first();

    if (!$guruBk) {
        return back()->with('error', 'Guru BK belum ditetapkan di kelas ini.');
    }

    LayananParenting::create([
        'siswa_id'     => $siswa->id,
        'guru_id'      => $guruBk->guru_id,
        'tanggal'      => $request->tanggal,
        'permasalahan' => $request->permasalahan,
        'tipe'         => 1,
    ]);

    return redirect()->route('siswa.parenting')->with('success', 'Layanan parenting berhasil dikirim.');
}

public function kehadiran()
{
    $user = Auth::user();

    if ($user->role !== 'siswa') {
        abort(403, 'Akses ditolak: Anda bukan siswa.');
    }

    $kehadiranSiswa = KehadiranSiswa::with([
            'bulan' => function ($q) {
                $q->withTrashed()->with([
                    'kelas' => function ($kelas) {
                        $kelas->withTrashed()->with(['tahunAjaran', 'semester']);
                    }
                ]);
            }
        ])
        ->where('siswa_id', $user->id)
        ->orderByDesc('id')
        ->get();

    return view('backend.siswa.kehadiran', compact('kehadiranSiswa'));
}


}