<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Guru;
use App\Models\LayananKonseling;
use App\Models\TahunAjaran;
use App\Models\Kelas;
use App\Models\KelasMapel;
use App\Models\Nilai;
use App\Models\JadwalLayanan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use App\Exports\SiswaNilaiExport;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\SiswaNilaiImport;
use App\Models\LayananParenting;

class GuruController extends Controller
{

public function index() 
{
    $user = Auth::user();

    // Cek apakah user memiliki data guru
    if (!$user->guru) {
        return view('backend.guru.index', [
            'totalKelas' => 0,
            'user' => $user
        ]);
    }

    $guru = $user->guru;

    // Cek apakah guru memiliki guru_id
    if (!$guru->guru_id) {
        return view('backend.guru.index', [
            'totalKelas' => 0,
            'user' => $user
        ]);
    }

    // Ambil semua kelasMapel yang diampu oleh guru
    $kelasMapelIds = KelasMapel::where('guru_id', $guru->guru_id)
        ->pluck('kelas_id')
        ->unique();

    // Jika guru belum memiliki kelas mapel, return dengan totalKelas = 0
    if ($kelasMapelIds->isEmpty()) {
        return view('backend.guru.index', [
            'totalKelas' => 0,
            'user' => $user
        ]);
    }

    $allKelas = Kelas::whereIn('id', $kelasMapelIds)
        ->with('tahunAjaran')
        ->get();

    // Jika tidak ada kelas yang ditemukan, return dengan totalKelas = 0
    if ($allKelas->isEmpty()) {
        return view('backend.guru.index', [
            'totalKelas' => 0,
            'user' => $user
        ]);
    }

    // Urutkan berdasarkan tahun ajaran terbaru
    $sortedKelas = $allKelas->sortByDesc(function ($k) {
        return $k->tahunAjaran->tahun_awal ?? 0;
    });

    // Ambil tahun ajaran terbaru dengan pengecekan null
    $kelasWithTahunAjaran = $sortedKelas->first();
    $tahunAjaranTerbaruId = $kelasWithTahunAjaran && $kelasWithTahunAjaran->tahunAjaran 
        ? $kelasWithTahunAjaran->tahun_ajaran_id 
        : null;

    // Jika tidak ada tahun ajaran terbaru, return dengan totalKelas = 0
    if (!$tahunAjaranTerbaruId) {
        return view('backend.guru.index', [
            'totalKelas' => 0,
            'user' => $user
        ]);
    }

    // Filter hanya kelas di tahun ajaran terbaru
    $kelasTahunTerbaru = $allKelas->filter(function ($kelas) use ($tahunAjaranTerbaruId) {
        return $kelas->tahun_ajaran_id == $tahunAjaranTerbaruId;
    });

    // Hitung hanya kelas unik berdasarkan nama dan tahun ajaran
    $uniqueKelas = $kelasTahunTerbaru->unique(function ($kelas) {
        return $kelas->nama . '-' . $kelas->tahun_ajaran_id;
    });

    $totalKelas = $uniqueKelas->count();

    return view('backend.guru.index', compact('totalKelas', 'user'));
}

public function penilaian(Request $request)
{
    $guru = Auth::user()->guru;
    $kelasParam = $request->kelas;

    $kelasMapelIds = KelasMapel::where('guru_id', $guru->guru_id)
        ->pluck('kelas_id')
        ->unique();

    $allKelas = Kelas::whereIn('id', $kelasMapelIds)
        ->with('tahunAjaran')
        ->get();

    $allKelas = $allKelas->sortByDesc(function ($k) {
        return $k->tahunAjaran->tahun_awal ?? 0;
    });

    $tahunAjaranTerbaru = optional($allKelas->first())->tahun_ajaran_id;

    $dropdownKelas = $allKelas->filter(function ($kelas) use ($tahunAjaranTerbaru) {
        return $kelas->tahun_ajaran_id == $tahunAjaranTerbaru;
    })->unique(function ($item) {
        return $item->nama . '-' . $item->tahun_ajaran_id;
    });

    $kelases = collect();
    $selectedKey = null;

    if ($kelasParam) {
        if (is_numeric($kelasParam)) {
            $kelas = Kelas::find($kelasParam);
            $nama = $kelas->nama;
            $tahunAjaranId = $kelas->tahun_ajaran_id;
            $selectedKey = $nama . '-' . $tahunAjaranId;
        } else {
            [$nama, $tahunAjaranId] = explode('-', $kelasParam);
            $selectedKey = $kelasParam;
        }

        $kelases = Kelas::with([
            'semester',
            'tahunAjaran',
            'kelasMapel' => function($q) use ($guru) {
                $q->where('guru_id', $guru->guru_id)
                  ->with(['mapel', 'nilai.siswa']);
            },
            'siswaKelas.siswa.user',
        ])
        ->where('nama', $nama)
        ->where('tahun_ajaran_id', $tahunAjaranId)
        ->whereIn('id', $kelasMapelIds)
        ->orderByDesc('semester_id')
        ->get();
    }

    if ($kelases->isEmpty() || $kelases->every(fn($k) => $k->kelasMapel->isEmpty())) {
        return view('backend.guru.penilaian', [
            'guru' => $guru,
            'dropdownKelas' => $dropdownKelas,
            'kelases' => collect(),
            'selectedKey' => $selectedKey,
        ]);
    }

    return view('backend.guru.penilaian', compact(
        'guru', 'dropdownKelas', 'kelases', 'selectedKey'
    ));
}

public function updateNilai(Request $request)
{
    $data = $request->validate([
      'kelas_mapel_id' => 'required|exists:kelas_mapel,id',
      'siswa_id'       => 'required|exists:siswa,siswa_id',
      'nilai'          => 'required|array',
      'nilai.*'        => 'nullable|integer|min:0|max:100',
    ]);

    $nilai = Nilai::where('kelas_mapel_id', $data['kelas_mapel_id'])
                  ->where('siswa_id', $data['siswa_id'])
                  ->firstOrFail();

    $raw = $nilai->s_nilai;
    $assoc = is_string($raw) ? json_decode($raw, true) : ($raw ?? []);

    foreach ($data['nilai'] as $idx => $val) {
        $labels = array_keys($assoc);
        $key = $labels[$idx] ?? null;
        if ($key) {
          $assoc[$key] = $val;
        }
    }

    $nilai->s_nilai = $assoc;
    $nilai->save();

    return back()->with('success', 'Nilai siswa berhasil diperbarui');
}

    public function updateJumlahSumatif(Request $request)
{
    $data = $request->validate([
        'kelas_mapel_id' => 'required|exists:kelas_mapel,id',
        'count'          => 'required|integer|min:1|max:10',
        'kelas'          => 'nullable|string',
    ]);

    $km = KelasMapel::with('nilai')->findOrFail($data['kelas_mapel_id']);
    if ($km->guru_id !== Auth::user()->guru->guru_id) {
        abort(403);
    }

    foreach ($km->nilai as $nilai) {
        $raw = $nilai->s_nilai;
        $arr = is_string($raw) ? json_decode($raw, true) : ($raw ?? []);
        $newArr = array_slice($arr, 0, $data['count']);
        $newArr = array_pad($newArr, $data['count'], null);
        $nilai->s_nilai = $newArr;
        $nilai->save();
    }

    return redirect()
        ->route('backend.guru.penilaian', ['kelas' => $data['kelas']])
        ->with('success', 'Jumlah sumatif diperbarui');
}

public function updateLabelSumatif(Request $request)
{
    $data = $request->validate([
        'kelas_mapel_id' => 'required|exists:kelas_mapel,id',
        'labels'         => 'required|array',
        'labels.*'       => 'required|string',
        'kelas'          => 'nullable|string',
    ]);

    $km = KelasMapel::with('nilai')->findOrFail($data['kelas_mapel_id']);
    if ($km->guru_id !== Auth::user()->guru->guru_id) abort(403);

    foreach ($km->nilai as $nilai) {

        // decode nilai lama jadi array asosiatif
        $oldAssoc = is_string($nilai->s_nilai)
                    ? json_decode($nilai->s_nilai, true)
                    : ($nilai->s_nilai ?? []);

        // ambil values-nya, sesuai urutan index
        $oldValues = array_values($oldAssoc);

        // s_nilai baru sesuai labels dan posisi value lama
        $new = [];
        foreach ($data['labels'] as $idx => $label) {
            $new[$label] = $oldValues[$idx] ?? null;
        }

        $nilai->s_nilai = $new;
        $nilai->save();
    }

    return redirect()
        ->route('backend.guru.penilaian', ['kelas' => $data['kelas']])
        ->with('success', 'Label Sumatif berhasil diubah');
}


public function exportNilai($kelas_mapel_id)
{
    $guruId = Auth::user()->guru->guru_id;
    $km = KelasMapel::findOrFail($kelas_mapel_id);
    abort_unless($km->guru_id === $guruId, 403);

    $fileName = "NilaiSiswa_{$km->mapel->nama}_Kelas-{$km->kelas->nama}_semester-{$km->kelas->semester->semester}_{$km->kelas->tahunAjaran->tahun_awal}-{$km->kelas->tahunAjaran->tahun_akhir}.xlsx";
    return Excel::download(
        new SiswaNilaiExport($kelas_mapel_id),
        $fileName
    );
}

public function importNilai(Request $request, $kelasMapelId)
{
    $request->validate([
        'file' => 'required|file|mimes:xlsx,xls',
    ]);

    try {
        Excel::import(new SiswaNilaiImport($kelasMapelId), $request->file('file'));
        return back()->with('success', 'Import nilai siswa berhasil.');
    } catch (\Exception $e) {
        return back()->withErrors(['msg' => 'Gagal import nilai: ' . $e->getMessage()]);
    }
}
    
public function profil()
{
    $user = auth()->user();

    if (!$user) {
        abort(403, 'Anda belum login.');
    }

    $guru = Guru::with('user')->where('guru_id', $user->id)->first();

    if (!$guru) {
        abort(404, 'Data guru tidak ditemukan.');
    }

    $tahunAjaranTerbaru = TahunAjaran::orderByDesc('tahun_awal')->first();

    $waliKelas = Kelas::where('wali_kelas_id', $user->id)
        ->whereHas('semester', function ($query) use ($tahunAjaranTerbaru) {
            $query->where('tahun_ajaran_id', $tahunAjaranTerbaru->id);
        })
        ->with('semester.tahunAjaran')
        ->first();

    $mapelDiampu = KelasMapel::with('mapel', 'kelas.semester.tahunAjaran')
        ->where('guru_id', $guru->guru_id)
        ->whereHas('kelas.semester', function ($query) use ($tahunAjaranTerbaru) {
            $query->where('tahun_ajaran_id', $tahunAjaranTerbaru->id);
        })
        ->first();

    return view('backend.guru.profil', compact('guru', 'waliKelas', 'mapelDiampu'));
}
    
public function waliKelas()
{
    $guru = Auth::user()->guru;

    // Ambil semua kelas yg dipegang guru sbg wali
    $kelasSemua = Kelas::where('wali_kelas_id', $guru->guru_id)
    ->with('tahunAjaran')
    ->get()
    ->sortByDesc(fn($k) => $k->tahunAjaran->tahun_awal ?? 0);

    if ($kelasSemua->isEmpty()) {
        return view('backend.guru.wali_kelas', ['kelases' => collect()]);
    }

    $tahunAjaranTerbaru = optional($kelasSemua->first())->tahun_ajaran_id;

    // Filter ulang hanya kelas dari tahun ajaran terbaru
    $kelases = $kelasSemua
                ->where('tahun_ajaran_id', $tahunAjaranTerbaru)
                ->sortByDesc('semester_id')
                ->values();

    $kelases->load([
        'semester',
        'tahunAjaran',
        'kelasMapel.mapel',
        'kelasMapel.guru.user',
        'kelasMapel.nilai',
        'siswaKelas.siswa.user',
    ]);

    return view('backend.guru.wali_kelas', compact('kelases'));
}

public function layanan()
{
    $guru = Auth::user()->guru;

    $layanans = LayananKonseling::with([
            'siswa' => function ($query) {
                $query->withTrashed()->with(['user' => fn ($q) => $q->withTrashed()]);
            },
            'jadwal' => function ($query) {
                $query->withTrashed();
            }
        ])
        ->whereHas('jadwal', function ($q) use ($guru) {
            $q->withTrashed()->where('guru_id', $guru->guru_id);
        })
        ->latest()
        ->get();

    foreach ($layanans as $layanan) {
        $layanan->sudah_ada_parenting = LayananParenting::where('siswa_id', $layanan->siswa->user->id ?? null)
            ->where('tanggal', $layanan->tanggal)
            ->exists();
    }

    $layanans_parenting = LayananParenting::with([
            'siswa' => fn ($q) => $q->withTrashed()->with(['user' => fn ($u) => $u->withTrashed()]),
            'guru' => fn ($q) => $q->withTrashed()->with(['user' => fn ($u) => $u->withTrashed()])
        ])
        ->where('guru_id', $guru->guru_id)
        ->where('tipe', 1)
        ->latest()
        ->get();

    return view('backend.guru.layanan', compact('layanans', 'layanans_parenting'));
}


public function storeHasilLayanan(Request $request)
    {
        $request->validate([
            'siswa_id'     => 'required|exists:siswa,siswa_id',
            'guru_id'      => 'required|exists:users,id',
            'tanggal'      => 'required|date',
            'tipe'         => 'required|in:0,1',
            'permasalahan' => 'required|string',
            'penyelesaian' => 'required|string',
            'hasil'        => 'required|string',
        ]);

        try {

            $data = $request->only([
                'siswa_id',
                'guru_id',
                'tanggal',
                'tipe',
                'permasalahan',
                'penyelesaian',
                'hasil',
            ]);

            LayananParenting::create($data);

            return back()->with('success', 'Hasil layanan berhasil dikirim.');
        } catch (\Exception $e) {
            return back()->with('error', 'Gagal simpan: ' . $e->getMessage());
        }
    }

public function updateLayanan(Request $request, $id)
{
    $layanan = LayananKonseling::findOrFail($id);
    $layanan->status = $request->status;
    $layanan->save();

    return redirect()->back()->with('success', 'Status layanan berhasil diupdate.');
}

    public function jadwalLayanan()
    {
        $guru = Auth::user()->guru;
        $jadwals = JadwalLayanan::where('guru_id', $guru->guru_id)
                ->orderBy('created_at', 'desc')
                ->get();

        return view('backend.guru.jadwal_layanan', compact('jadwals'));
    }

    public function storeJadwal(Request $request)
    {
        $request->validate([
            'hari' => 'required|string|max:20',
            'jam'  => 'required|string|max:50',
        ]);

        JadwalLayanan::create([
            'guru_id' => Auth::user()->guru->guru_id,
            'hari'    => $request->hari,
            'jam'     => $request->jam,
        ]);

        return redirect()->route('backend.guru.jadwalLayanan')->with('success', 'Jadwal Layanan berhasil ditambahkan.');
    }

public function updateJadwal(Request $request, $id)
{
    $request->validate([
        'hari' => 'required|string|max:20',
        'jam'  => 'required|string|max:50',
    ]);

    $jadwal = JadwalLayanan::findOrFail($id);

    if ($jadwal->guru_id !== Auth::user()->guru->guru_id) {
        abort(403);
    }

    // Soft delete data lama
    $jadwal->delete();

    // Buat data baru
    JadwalLayanan::create([
        'guru_id'  => $jadwal->guru_id,
        'siswa_id' => $jadwal->siswa_id,
        'hari'     => $request->hari,
        'jam'      => $request->jam,
    ]);

    return redirect()->route('backend.guru.jadwalLayanan')
        ->with('success', 'Jadwal Layanan berhasil diperbarui.');
}


    public function deleteJadwal($id)
    {
        $jadwal = JadwalLayanan::findOrFail($id);

        if ($jadwal->guru_id !== Auth::user()->guru->guru_id) {
            abort(403);
        }

        $jadwal->delete();

        return redirect()->back()->with('success', 'Jadwal Layanan berhasil dihapus.');
    }
}